/**
 * \file: grl_os_abstraction_user.h
 *
 * \release: $Name: SVG_r2_B30_RC2 $
 *
 * System call abstraction function.
 *
 * \component: svg_common
 *
 * \author: CSpetzler
 *
 * \copyright: (c) 2003 - 2010 ADIT Corporation
 *
 */


#include <stdlib.h>
#include <string.h>
#include <malloc.h>
#include <stdio.h>
#include <sys/time.h>
#include <grl_os_abstraction.h>
#include <svg_common.h>

#include <pthread.h>
#include <semaphore.h>
#include <fcntl.h>

typedef struct lock_id {
	sem_t * semaphore;
	SVGBoolean named_sem;
} * GRL_lock;

typedef struct event_flg {
	pthread_mutex_t mutex;
	pthread_cond_t cond_var;
	GRL_flg_ptn flag;

} GRL_evtflg;


/**
 * \func convert_add_times
 *
 * Converts time from timeval to timespec and add time in ms
 *
 * \in 	tp 			time struct timeval, set with gettimeofday()
 *		wai_time	timeout in ms
 *
 * \out	ts			updated time struct timespec, with added timeout specified by wai_time
 * \see
 */
static void convert_add_times(
                struct timespec* ts,
                struct timeval* tp,
                SVGInt32 wai_time)
{
	ts->tv_sec = tp->tv_sec + wai_time / 1000;
	ts->tv_nsec = tp->tv_usec * 1000;
	ts->tv_nsec += (wai_time % 1000) * 1000000;
	if (ts->tv_nsec > 1000000000) {
		ts->tv_nsec = ts->tv_nsec % 1000000000;
		ts->tv_sec += 1;
	}
}

/**
 * \func GRL_cre_lock
 *
 * Creates a lock. When dsname name is set the lock is shared between processes.
 * If not set the lock is only usable within one process.
 *
 * \return error
 *
 * \see
 */

GRL_error GRL_cre_lock(GRL_lock_id *p_lock_id, SVGChar *dsname)
{
/* PRQA: Lint Message 593: deactivation lock is used system wide, so it does not have to be freed */
/*lint -e593 */
	GRL_error return_val = GRL_NO_ERROR;
	GRL_lock lock_id = NULL;
	if (p_lock_id != NULL) {
		lock_id = malloc(sizeof(*lock_id));
		if (lock_id == NULL) {
			SVG_COM_E("SVG_OUT_OF_MEMORY IN GRL_CRE_LOCK");
			return_val = SVG_OUT_OF_MEMORY;
			return return_val;
		}
		/*use a process only semaphore*/
		if (dsname == NULL) {
			lock_id->named_sem = SVG_FALSE;
			lock_id->semaphore = malloc(sizeof(sem_t));
			return_val = sem_init(lock_id->semaphore, 0, 1);

		} else {
			/*use a named semaphore*/
			lock_id->named_sem = SVG_TRUE;
			lock_id->semaphore = sem_open(dsname, O_CREAT, 0777, 1);
		}

		if ((lock_id->semaphore == SEM_FAILED) || (return_val == -1)) {
			perror("GRL_cre_lock failed");
			return_val = SVG_NOT_INITIALIZED;
		} else {
			*p_lock_id = (GRL_lock_id)lock_id;
		}

	} else {
		SVG_COM_E("SVG_POINTER_NULL IN  GRL_CRE_LOCK");
		return_val = SVG_POINTER_NULL;
	}
	return return_val;
/*lint +e593 */
}

/**
 * \func GRL_del_lock
 *
 * Deletes a lock
 *
 * \return error
 *
 * \see
 */
GRL_error GRL_del_lock(GRL_lock_id lock_id)
{
	GRL_error return_val = GRL_NO_ERROR;

	if (lock_id != NULL) {
		GRL_lock lock = (GRL_lock)lock_id;
		if (lock->named_sem == SVG_TRUE) {
			return_val = sem_close(lock->semaphore);
		} else {
			return_val = sem_destroy(lock->semaphore);
			free(lock->semaphore);
		}

		if (return_val == -1) {
			return_val = SVG_INVALID_OPERATION;
			perror("GRL_del_lock failed");
		} else {
			free(lock);
		}

	} else {
		SVG_COM_E("Nullpointer as parameter in GRL_del_lock");
		return_val = SVG_POINTER_NULL;
	}

	return return_val;
}

/**
 * \func GRL_wai_lock
 *
 * Wait for a lock
 *
 * \return error
 *
 * \see
 */
GRL_error GRL_wai_lock(GRL_lock_id lock_id)
{
	GRL_error return_val = GRL_NO_ERROR;

	if (lock_id != NULL) {
		GRL_lock lock = (GRL_lock)lock_id;
		return_val = sem_wait(lock->semaphore);
		if (return_val == -1) {
			return_val = SVG_INVALID_OPERATION;
			perror("GRL_wai_lock failed");
		}
	} else {
		SVG_COM_E("Nullpointer as parameter in GRL_wai_lock");
		return_val = SVG_POINTER_NULL;
	}

	return return_val;
}

/**
 * \func GRL_wai_lock_tmo
 *
 * Wait for a lock with time out parameter
 *
 * \return error
 *
 * \see
 */
GRL_error GRL_wai_lock_tmo(GRL_lock_id lock_id, SVGInt32 tim)
{
	GRL_error return_val = GRL_NO_ERROR;
	struct timespec ts;
	struct timeval  tp;

	gettimeofday(&tp, NULL);
	return_val = gettimeofday(&tp, NULL);
	if (return_val == GRL_NO_ERROR) {
		/*convert and add time*/
		convert_add_times(&ts, &tp, tim);
		if (lock_id != NULL) {
			GRL_lock lock = (GRL_lock)lock_id;
			return_val = sem_timedwait(lock->semaphore, &ts);
			if (return_val == -1) {
				return_val = SVG_INVALID_OPERATION;
				perror("GRL_wai_lock_tmo failed");
			}
		} else {
			SVG_COM_E("Nullpointer as parameter in GRL_wai_lock_tmo");
			return_val = SVG_POINTER_NULL;
		}
	} else {
		perror("GRL_wai_lock_tmo failed");
		return_val = SVG_INVALID_OPERATION;
	}
	return return_val;
}

/**
 * \func GRL_sig_lock
 *timedwait
 * Unlock a lock (Semaphore)
 *
 * \return error
 *
 * \see
 */
GRL_error GRL_sig_lock(GRL_lock_id lock_id)
{
	GRL_error return_val = GRL_NO_ERROR;

	if (lock_id != NULL) {
		GRL_lock lock = (GRL_lock)lock_id;
		return_val = sem_post(lock->semaphore);
		if (return_val == -1) {
			return_val = SVG_INVALID_OPERATION;
			perror("GRL_wai_lock_tmo failed");
		}
	} else {
		SVG_COM_E("Nullpointer as parameter in GRL_sig_lock");
		return_val = SVG_POINTER_NULL;
	}

	return return_val;
}

/**
 * \func GRL_is_locked
 *
 * Checks if a lock is set (BLock).
 *
 * \return status {0,1}
 *         error < 0
 *
 * \see
 */
SVGBoolean GRL_is_locked(GRL_lock_id lock_id)
{
	SVGBoolean return_val = SVG_FALSE;
	SVGInt32 value;
	if (lock_id != NULL) {
		GRL_lock lock = (GRL_lock)lock_id;
		if (sem_getvalue(lock->semaphore, &value) != -1) {
			if (value > 0) {
				/*locked*/
				return_val = SVG_TRUE;
			} else {
				/*not locked*/
				return_val = SVG_FALSE;
			}
		} else {
			return_val = SVG_FALSE;
			perror("GRL_is_locked failed");
		}
	} else {
		SVG_COM_E("Nullpointer as parameter in GRL_is_locked");
		return_val = SVG_FALSE;
	}
	return return_val;
}

/*************************************************************************/
/*                                                                       */
/*                  EVENT FLAGS ABSTRACTION INTERFACE                    */
/*                                                                       */
/*************************************************************************/
static int cmp_flg_and(GRL_flg_ptn f, GRL_flg_ptn wf)
{
	int rc = -1;
	if ((f & wf) == wf)
		rc = 0;
	return rc;
}
static int cmp_flg_or(GRL_flg_ptn f, GRL_flg_ptn wf)
{
	int rc = -1;
	if ((f & wf) > 0)
		rc = 0;
	return rc;
}

GRL_error GRL_cre_flg(GRL_evtflg_id *evt_flg)
{
	GRL_error rc = GRL_NO_ERROR;
	GRL_evtflg *grl_evt_flg = 0;
	if (evt_flg != NULL) {
		grl_evt_flg = malloc(sizeof(GRL_evtflg));
		if (grl_evt_flg != NULL) {
			grl_evt_flg->flag = 0x00000000;
			rc = pthread_mutex_init(&grl_evt_flg->mutex, NULL);
			if (rc == 0) {
				rc = pthread_cond_init(&grl_evt_flg->cond_var, NULL);
				if (rc == 0) {
					/*everything OK*/
					*evt_flg = grl_evt_flg;
				} else {
					pthread_mutex_destroy(&grl_evt_flg->mutex);
					free(grl_evt_flg);
					printf("ERROR IN %s AT pthread_cond_init\n",
							__FUNCTION__);
				}
			} else {
				free(grl_evt_flg);
				printf("ERROR IN %s AT pthread_mutex_init\n", __FUNCTION__);
			}
		} else {
			SVG_COM_E("Failed to allocate memory in GRL_cre_flg");
			rc = SVG_OUT_OF_MEMORY;
		}
	} else {
		SVG_COM_E("Nullpointer as parameter in GRL_cre_flg");
		rc = SVG_POINTER_NULL;
	}
	return rc;
}

GRL_error GRL_del_flg(GRL_evtflg_id evt_flg)
{
	GRL_error rc = GRL_NO_ERROR;
	GRL_evtflg *grl_evt_flg = (GRL_evtflg *)evt_flg;
	if (grl_evt_flg != NULL) {
		rc = pthread_cond_destroy(&grl_evt_flg->cond_var);
		if (rc == GRL_NO_ERROR) {
			rc = pthread_mutex_destroy(&grl_evt_flg->mutex);
			free(grl_evt_flg);
			if (rc == GRL_NO_ERROR) {

			} else {
				printf("ERROR destroy mutex,rc = %d\n", rc);
			}
		} else {
			printf("ERROR destroy cond variable, rc = %d\n", rc);
		}
	} else {
		SVG_COM_E("Nullpointer as parameter in GRL_del_flg");
		rc = SVG_POINTER_NULL;
	}
	return rc;
}

GRL_error GRL_set_flg(GRL_evtflg_id evt_flg, GRL_flg_ptn ptrn)
{
/* PRQA: Lint Message 454,456: deactivation unlock is only allowed when lock was successful */
/*lint -e454 -e456 */
	GRL_error rc = GRL_NO_ERROR;
	GRL_evtflg *grl_evt_flg = (GRL_evtflg *)evt_flg;

	if (grl_evt_flg != NULL) {
		rc = pthread_mutex_lock(&grl_evt_flg->mutex);
		if (rc == GRL_NO_ERROR) {
			grl_evt_flg->flag |= ptrn;

			rc = pthread_cond_broadcast(&grl_evt_flg->cond_var);
			if (rc == GRL_NO_ERROR) {

			} else {
				printf("ERROR broadcast cond variable\n");
			}

			rc = pthread_mutex_unlock(&grl_evt_flg->mutex);
			if (rc != GRL_NO_ERROR) {
				printf("ERROR unlock mutex\n");
			}
		} else {
			printf("ERROR lock mutex\n");
		}
	} else {
		SVG_COM_E("Nullpointer as parameter in GRL_set_flg");
		rc = SVG_POINTER_NULL;
	}
	return rc;
/*lint +e454 +e456 */
}
/*************************************************************************
 *
 * <Function>strstr c
 *    GRL_wai_flg
 *
 * <Description>
 *    wait for the pattern .
 *
 * <Input>
 * 	evt_flg    :: pointer of the event flag
 *    ptrn       :: specified the pattern for waiting (matched with flag in evt_flg)
 *    wait_mod   :: specified or(FLG_CLR) or and(FLG_BITCLR) waiting of pattern bits
 * 			      specified also how to proceed with flag(in evt_flg) is pattern is matched
 * 			      clear all bits(FLG_CLR) or clear matched bits(FLG_BITCLR)
 * 			      example: wmode = FLG_WAI_OR |FLG_BITCLR
 *    r_flg_ptrn :: return the flag value in evt_flg after matching
 *    wai_time   :: specified the time to wait in ms
 *    			  -1 is infinity wait
 *    			   0 is no waiting just poll the flag in evt_flg		*/

GRL_error GRL_wai_flg(
		GRL_evtflg_id evt_flg,
                GRL_flg_ptn ptrn,
                GRL_flg_wai_mod wait_mode,
                GRL_flg_ptn *r_ptrn,
                SVGInt32 wai_time)
{
/* PRQA: Lint Message 454,456: deactivation unlock is only allowed when lock was successful */
/*lint -e454 -e456 */
	GRL_error rc = GRL_NO_ERROR;
	GRL_evtflg *grl_evt_flg = (GRL_evtflg *)evt_flg;

	struct timespec ts;
	struct timeval tp;
	int (*cmp_flg)(GRL_flg_ptn, GRL_flg_ptn);

	if (wait_mode & FLG_WAI_OR)/*decision or, and pattern comparison*/
		cmp_flg = cmp_flg_or;
	else
		cmp_flg = cmp_flg_and;

	if (grl_evt_flg != NULL) {
		rc = pthread_mutex_lock(&grl_evt_flg->mutex);
		if (rc == GRL_NO_ERROR) {
			if (wai_time > 0) /*timed wait*/
			{
				rc = gettimeofday(&tp, NULL);
				if (rc == GRL_NO_ERROR) { /*convert and add time*/
					convert_add_times(&ts, &tp, wai_time);
					while ((cmp_flg(grl_evt_flg->flag, ptrn) != 0)
							&& (rc == GRL_NO_ERROR)) {
						/*wait time or flag*/
						rc = pthread_cond_timedwait(
								&grl_evt_flg->cond_var,
								&grl_evt_flg->mutex, &ts);
					}
				} else {
					printf("ERROR IN %s AT gettimeofday\n",
							__FUNCTION__);
				}
			} else if (wai_time == 0)/*don*t wait just poll, time = 0*/
			{
				rc = cmp_flg(grl_evt_flg->flag, ptrn);
			} else /*infinity wait  , time = -1(negative)*/
			{
				while ((cmp_flg(grl_evt_flg->flag, ptrn) != 0) && (rc
						== GRL_NO_ERROR)) {
					rc = pthread_cond_wait(&grl_evt_flg->cond_var,
							&grl_evt_flg->mutex);
				}
			}
			if (rc == GRL_NO_ERROR) {
				*r_ptrn = grl_evt_flg->flag;
				if (wait_mode & FLG_BITCLR) {
					grl_evt_flg->flag &= ~ptrn;//clear matched bit
				}
				if (wait_mode & FLG_CLR) {
					grl_evt_flg->flag = 0;//clear all bits
				}
			} else if (rc > 0) {
				printf(
						"ERROR IN %s AT pthread_cond_timedwait( time over)\n",
						__FUNCTION__);
			} else {
				/*timeout occurred */
				//printf("ERROR IN %s pattern not matched ret code = %d wai time = %d\n ",__FUNCTION__,rc,wai_time);
			}
			pthread_mutex_unlock(&grl_evt_flg->mutex);
		} else {
			printf("ERROR IN %s AT pthread_mutex_lock, rc = %d\n",
					__FUNCTION__, rc);
		}
	} else {
		SVG_COM_E("Nullpointer as parameter in GRL_wai_flg");
		rc = SVG_POINTER_NULL;
	}
	return rc;
/*lint +e454 +e456 */
}

GRL_error GRL_clr_flg(GRL_evtflg_id evt_flg, GRL_flg_ptn clr_ptrn)
{
/* PRQA: Lint Message 454,456: deactivation unlock is only allowed when lock was successful */
/*lint -e454 -e456 */
	GRL_error rc = 0;
	GRL_evtflg *grl_evt_flg = (GRL_evtflg *)evt_flg;

	if (grl_evt_flg != NULL) {
		unsigned int temp = grl_evt_flg->flag | clr_ptrn;

		rc = pthread_mutex_lock(&grl_evt_flg->mutex);
		if (rc == GRL_NO_ERROR) {
			grl_evt_flg->flag = temp ^ clr_ptrn;

			rc = pthread_mutex_unlock(&grl_evt_flg->mutex);
			if (rc != GRL_NO_ERROR) {
				printf("ERROR unlock mutex\n");
			}
		} else {
			printf("ERROR lock mutex\n");
		}
	} else {
		SVG_COM_E("Nullpointer as parameter in GRL_clr_flg");
		rc = SVG_POINTER_NULL;
	}
	return rc;
/*lint +e454 +e456 */
}

/*************************************************************************/
/*                                                                       */
/*                      MEMORY ABSTRACTION INTERFACE                     */
/*                                                                       */
/*************************************************************************/

/*************************************************************************/
/*                                                                       */
/* <Function>                                                            */
/*    GRL_malloc_1D_resource                                                         */
/*                                                                       */
/* <Description>                                                         */
/*    The memory allocation function.                                    */
/*                                                                       */
/* <Input>                                                               */
/*    memory :: A pointer to the memory object.                          */
/*                                                                       */
/*    size   :: The requested size in bytes.                             */
/*                                                                       */
/* <Return>                                                              */
/*    The address of newly allocated block.                              */
/*                                                                       */
void*
GRL_malloc_1D_resource(long size)
{
	return malloc( size );
}

/*************************************************************************/
/*                                                                       */
/* <Function>                                                            */
/*    GRL_realloc_1D_resource                                                        */
/*                                                                       */
/* <Description>                                                         */
/*    The memory reallocation function.                                  */
/*                                                                       */
/* <Input>                                                               */
/*    memory   :: A pointer to the memory object.                        */
/*                                                                       */
/*    cur_size :: The current size of the allocated memory block.        */
/*                                                                       */
/*    new_size :: The newly requested size in bytes.                     */
/*                                                                       */
/*    block    :: The current address of the block in memory.            */
/*                                                                       */
/* <Return>                                                              */
/*    The address of the reallocated memory block.                       */
/*                                                                       */
void*
GRL_realloc_1D_resource(long cur_size, long new_size, void* block)
{
	cur_size = cur_size;/*getting rid of compiler warning: don't use*/
	return realloc( block, new_size );
}

/*************************************************************************/
/*                                                                       */
/* <Function>                                                            */
/*    GRL_free_1D_resource                                               */
/*                                                                       */
/* <Description>                                                         */
/*    The memory release function.                                       */
/*                                                                       */
/* <Input>                                                               */
/*    memory  :: A pointer to the memory object.                         */
/*                                                                       */
/*    block   :: The address of block in memory to be freed.             */
/*                                                                       */
void GRL_free_1D_resource(void* block)
{
	free( block );
}

